<?php
/**
 * File used to include ajax actions.
 *
 * @package live-news
 */

/**
 * This class should be used to include ajax actions.
 */
class Daln_Ajax {

	/**
	 * The single instance of the class.
	 *
	 * @var null
	 */
	protected static $instance = null;

	/**
	 * The single instance of the shared class.
	 *
	 * @var Daln_Shared|null
	 */
	private $shared = null;

	/**
	 * Return an instance of this class.
	 *
	 * @return self|null
	 */
	public static function get_instance() {

		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	private function __construct() {

		// Assign an instance of the plugin info.
		$this->shared = Daln_Shared::get_instance();

		// Ajax requests --------------------------------------------------------.
		add_action( 'wp_ajax_set_status_cookie', array( $this, 'set_status_cookie' ) );
		add_action( 'wp_ajax_nopriv_set_status_cookie', array( $this, 'set_status_cookie' ) );

		add_action( 'wp_ajax_get_ticker_data', array( $this, 'get_ticker_data' ) );
		add_action( 'wp_ajax_nopriv_get_ticker_data', array( $this, 'get_ticker_data' ) );

		add_action( 'wp_ajax_update_default_colors', array( $this, 'update_default_colors' ) );
	}

	/**
	 * Set the cookie used to determine the status (open or closed) of the news ticker. This request is triggered when
	 * the used clicks on the open or close button.
	 *
	 * @return void
	 */
	public function set_status_cookie() {

		// Check the referer.
		check_ajax_referer( 'live-news', 'security' );

		// Save the current status ( open/closed ) in a cookie.
		if ( isset( $_POST['status'] ) ) {

			if ( 'open' === $_POST['status'] ) {

				setcookie( 'live_news_status', 'open', 0, '/' );

			} else {

				setcookie( 'live_news_status', 'closed', 0, '/' );

			}
		}

		echo 'success';

		die();
	}

	/**
	 * Generate an XML response with included all the data of the ticker. The data are generated based on the options
	 * defined for the specific ticker.
	 *
	 * @return void
	 * @throws Exception
	 */
	public function get_ticker_data() {

		// Check the referer.
		check_ajax_referer( 'live-news', 'security' );

		// Get the ticker id.
		$ticker_id = isset( $_POST['ticker_id'] ) ? intval( $_POST['ticker_id'], 10 ) : null;

		// Get the ticker information.
		global $wpdb;
		$table_name = $wpdb->prefix . $this->shared->get( 'slug' ) . '_tickers';
		$safe_sql   = $wpdb->prepare( "SELECT * FROM $table_name WHERE id = %d", $ticker_id );
		$ticker_obj = $wpdb->get_row( $safe_sql );

		// If there isn't a ticker associated with this ticker_id die().
		if ( null === $ticker_obj ) {
			die( 'Invalid Ticker ID.' );
		}

		// START OUTPUT.

		// Generate the xml header.
		header( 'Content-type: text/xml' );
		header( 'Pragma: public' );
		header( 'Cache-control: private' );
		header( 'Expires: -1' );

		// Get the transient with included the data of the ticker if available.
		$outstr = get_transient( 'daln_ticker_' . $ticker_obj->id );

		// Generate the data of the ticker only if the transient with the data is not available.
		if ( false === $outstr ) {

			$outstr = '<?xml version="1.0" encoding="UTF-8" ?>';

			$outstr .= '<ticker>';

			// generate featured news XML ---------------------------------------------------------------------------.
			$outstr .= '<featurednews>';

			switch ( $ticker_obj->source ) {

				// Enter the news manually.
				case 1:
					global $wpdb;
					$table_name = $wpdb->prefix . $this->shared->get( 'slug' ) . '_featured_news';
					$results    = $wpdb->get_results( "SELECT id, news_title, news_excerpt, url FROM $table_name WHERE ticker_id = $ticker_obj->id ORDER BY id DESC LIMIT 1", ARRAY_A );

					if ( count( $results ) > 0 ) {
						foreach ( $results as $result ) {

							$outstr .= '<news>';
							$outstr .= '<newstitle>' . esc_attr( $this->shared->strlen_no_truncate( stripslashes( $result['news_title'] ), $ticker_obj->featured_title_maximum_length ) ) . '</newstitle>';
							$outstr .= '<newsexcerpt>' . esc_attr( $this->shared->strlen_no_truncate( stripslashes( $result['news_excerpt'] ), $ticker_obj->featured_excerpt_maximum_length ) ) . '</newsexcerpt>';
							$outstr .= '<url>' . esc_attr( stripslashes( $result['url'] ) ) . '</url>';
							$outstr .= '</news>';

						}
					}
					break;

				// Get the news automatically from the WordPress posts.
				case 2:
					$args = array(
						'numberposts' => '1',
						'offset'      => '0',
						'orderby'     => 'date',
						'order'       => 'DESC',
						'category'    => $ticker_obj->category,
					);

					$myposts = get_posts( $args );
					foreach ( $myposts as $post ) {

						setup_postdata( $post );

						$outstr .= '<news>';
						$outstr .= '<newstitle>' . esc_attr( $this->shared->strlen_no_truncate( stripslashes( $post->post_title ), $ticker_obj->featured_title_maximum_length ) ) . '</newstitle>';
						$outstr .= '<newsexcerpt>' . esc_attr( $this->shared->strlen_no_truncate( stripslashes( $post->post_excerpt ), $ticker_obj->featured_excerpt_maximum_length ) ) . '</newsexcerpt>';
						$outstr .= '<url>' . esc_attr( stripslashes( get_permalink( $post->ID ) ) ) . '</url>';
						$outstr .= '</news>';

					}
					break;

				// Get the news from a specified feed RSS.
				case 3:
					$rss = simplexml_load_file( $ticker_obj->url_rss );

					// loop through the items.
					$counter = 0;
					foreach ( $rss->channel->item as $single_item ) {

						// Use only the first entries.
						if ( $counter > 0 ) {
							break;
						}
						++$counter;

						$outstr .= '<news>';
						$outstr .= '<newstitle>' . esc_attr( $this->shared->strlen_no_truncate( $single_item->title, $ticker_obj->featured_title_maximum_length ) ) . '</newstitle>';
						$outstr .= '<newsexcerpt>' . esc_attr( $this->shared->strlen_no_truncate( $single_item->description, $ticker_obj->featured_excerpt_maximum_length ) ) . '</newsexcerpt>';
						$outstr .= '<url>' . esc_attr( $single_item->link ) . '</url>';
						$outstr .= '</news>';

					}

					break;

			}

			$outstr .= '</featurednews>';

			// Generate sliding news XML ------------------------------------------------------------------------------.
			$outstr .= '<slidingnews>';

			// Get number of sliding news from the option.
			$number_of_sliding_news = intval( $ticker_obj->number_of_sliding_news, 10 );

			/*
			 * Set the offset based on the "Hide Featured News" option. If the featured news is hidden then offset is 0,
			 * if the featured news is shown the offset is 1.
			 */
			if ( 2 === $ticker_obj->hide_featured_news ) {
				$offset = 0;
			} else {
				$offset = 1;
			}

			switch ( $ticker_obj->source ) {

				// Enter the news manually.
				case 1:
					global $wpdb;
					$table_name = $wpdb->prefix . $this->shared->get( 'slug' ) . '_sliding_news';
					$results    = $wpdb->get_results( "SELECT id, news_title, url, text_color, text_color_hover, background_color, background_color_opacity, image_before, image_after FROM $table_name WHERE ticker_id = $ticker_obj->id ORDER BY id DESC LIMIT $number_of_sliding_news", ARRAY_A );

					if ( count( $results ) > 0 ) {
						foreach ( $results as $result ) {

							$outstr .= '<news>';
							$outstr .= '<newstitle>' . esc_attr( $this->shared->strlen_no_truncate( stripslashes( $result['news_title'] ), $ticker_obj->sliding_news_maximum_length ) ) . '</newstitle>';
							$outstr .= '<url>' . esc_attr( stripslashes( $result['url'] ) ) . '</url>';
							$outstr .= '<text_color>' . esc_attr( stripslashes( $result['text_color'] ) ) . '</text_color>';
							$outstr .= '<text_color_hover>' . esc_attr( stripslashes( $result['text_color_hover'] ) ) . '</text_color_hover>';
							$outstr .= '<background_color>' . esc_attr( stripslashes( $result['background_color'] ) ) . '</background_color>';
							$outstr .= '<background_color_opacity>' . esc_attr( $result['background_color_opacity'] ) . '</background_color_opacity>';
							$outstr .= '<image_before>' . esc_attr( stripslashes( $result['image_before'] ) ) . '</image_before>';
							$outstr .= '<image_after>' . esc_attr( stripslashes( $result['image_after'] ) ) . '</image_after>';
							$outstr .= '</news>';

						}
					}

					break;

				// Get the news automatically from the WordPress posts.
				case 2:
					$args = array(
						'numberposts' => $number_of_sliding_news,
						'offset'      => $offset,
						'orderby'     => 'date',
						'order'       => 'DESC',
						'category'    => $ticker_obj->category,
					);

					$myposts = get_posts( $args );
					foreach ( $myposts as $post ) {

						setup_postdata( $post );

						$outstr .= '<news>';
						$outstr .= '<newstitle>' . esc_attr( $this->shared->strlen_no_truncate( stripslashes( $post->post_title ), $ticker_obj->sliding_news_maximum_length ) ) . '</newstitle>';
						$outstr .= '<url>' . esc_attr( stripslashes( get_permalink( $post->ID ) ) ) . '</url>';
						$outstr .= '</news>';
					}
					break;

				// Get the news from a specified feed RSS.
				case 3:
					$rss = simplexml_load_file( $ticker_obj->url_rss );

					// Loop through the items.
					$counter = 0;
					foreach ( $rss->channel->item as $single_item ) {

						// Skip the first item if the offset is set to 1.
						++$counter;
						if ( 1 === $offset && 1 === $counter || ( ( $counter - $offset ) > $number_of_sliding_news ) ) {
							continue;
						}

						$outstr .= '<news>';
						$outstr .= '<newstitle>' . esc_attr( $this->shared->strlen_no_truncate( $single_item->title, $ticker_obj->sliding_news_maximum_length ) ) . '</newstitle>';
						$outstr .= '<url>' . esc_attr( $single_item->link ) . '</url>';
						$outstr .= '</news>';

					}

					break;

				// Get the news from Twitter.
				case 4:
					require_once $this->shared->get( 'dir' ) . 'public/inc/twitter-api-php/TwitterAPIExchange.php';

					$settings = array(
						'oauth_access_token'        => stripslashes( $ticker_obj->twitter_oauth_access_token ),
						'oauth_access_token_secret' => stripslashes( $ticker_obj->twitter_oauth_access_token_secret ),
						'consumer_key'              => stripslashes( $ticker_obj->twitter_consumer_key ),
						'consumer_secret'           => stripslashes( $ticker_obj->twitter_consumer_secret ),
					);

					$request_method = 'GET';
					$url            = 'https://api.twitter.com/1.1/statuses/user_timeline.json';
					$getfield       = '?screen_name=' . stripslashes( $ticker_obj->twitter_username ) . '&count=' . $number_of_sliding_news . '&tweet_mode=extended';

					$twitter = new TwitterAPIExchange( $settings );

					$user_timeline_json = $twitter->setGetfield( $getfield )->
					buildOauth( $url, $request_method )->
					performRequest();

					$user_timeline = json_decode( $user_timeline_json );

					// Loop through the items.
					foreach ( $user_timeline as $key => $single_tweet ) {

						// Remove the specified elements from the tweet (links, hashtags, usernames).
						$newstitle = $this->shared->twitter_remove_elements( $single_tweet->full_text, $ticker_obj->twitter_strip_links, $ticker_obj->twitter_strip_hashtags, $ticker_obj->twitter_strip_usernames );

						// Limit the maximum number of characters of the tweet if specified with the "Sliding News Maximum Length" option.
						$newstitle = $this->shared->strlen_no_truncate( $newstitle, $ticker_obj->sliding_news_maximum_length );

						$outstr .= '<news>';
						$outstr .= '<newstitle>' . esc_attr( $newstitle ) . '</newstitle>';
						$outstr .= '<url>' . esc_attr( 'https://twitter.com/' . stripslashes( $ticker_obj->twitter_username ) . '/status/' . $single_tweet->id_str ) . '</url>';
						$outstr .= '</news>';

					}

					break;

			}

			$outstr .= '</slidingnews>';

			// generate current time XML ------------------------------------------------------------------------------.
			$current_time = current_time( 'timestamp' ) + $ticker_obj->clock_offset;

			$outstr .= '<time>' . esc_attr( stripslashes( $current_time ) ) . '</time>';

			$outstr .= '</ticker>';

			if ( $ticker_obj->transient_expiration > 0 ) {
				set_transient( 'daln_ticker_' . $ticker_obj->id, $outstr, $ticker_obj->transient_expiration );
			}
		}

		echo $outstr;

		die();
	}

	/**
	 * Retrieve the "Sliding News Color", the "Sliding News Color Hover, and the "Sliding News Background Color" from
	 * the tickers to initialize the values of the three fields in the "Sliding News" menu.
	 *
	 * @return void
	 */
	public function update_default_colors() {

		// Check the referer.
		check_ajax_referer( 'live-news', 'security' );

		// Check the capability.
		if ( ! current_user_can( get_option( $this->shared->get( 'slug' ) . '_sliding_menu_capability' ) ) ) {
			die();
		}

		// Get the missing word id.
		$ticker_id = isset( $_POST['ticker_id'] ) ? intval( $_POST['ticker_id'], 10 ) : null;

		// Get the ticker data.
		global $wpdb;
		$table_name = $wpdb->prefix . $this->shared->get( 'slug' ) . '_tickers';
		$safe_sql   = $wpdb->prepare( "SELECT sliding_news_color, sliding_news_color_hover, sliding_news_background_color FROM $table_name WHERE id = %d ", $ticker_id );
		$ticker_obj = $wpdb->get_row( $safe_sql );

		// Remove the slashes before sending the json response.
		$response                                = new stdClass();
		$response->sliding_news_color            = stripslashes( $ticker_obj->sliding_news_color );
		$response->sliding_news_color_hover      = stripslashes( $ticker_obj->sliding_news_color_hover );
		$response->sliding_news_background_color = stripslashes( $ticker_obj->sliding_news_background_color );

		// Return the data with json.
		echo wp_json_encode( $response );

		die();
	}
}
